/**
 * \file: grl_bitmap_decoder_util.c
 *
 * version: $Id: grl_bitmap_decoder_util.c,v 1.12 2010/02/10 14:39:26 tkniep Exp $
 *
 * Implementation of the utility functions which implement common functions
 * that are used by the decoder modules.
 *
 * \component: SVG Bitmap Decoder (SVGBMPDEC)
 *
 * \author: T. Kniep (tkniep@de.adit-jv.com)
 *
 * \copyright: (c) 2009 ADIT Corporation
 *
 ***********************************************************************/

#include <svg_typedef.h>
#include <svg_error.h>
#include <grl_configuration.h>
#include "svg_bitmap_decoder.h"
#include "svg_bitmap_decoder_err.h"
#include "grl_bitmap_decoder_conf.h"
#include "grl_bitmap_decoder_util.h"


/*******************************************************************************************
 *   Global variables
 *******************************************************************************************/

/**
 * Global configuration structure. It is only written during initialization with values from
 * the devconf. 
 */
static GRL_BMPDEC_conf      grl_bmpdec_conf;


/*******************************************************************************************
 *   Function implementations
 *******************************************************************************************/

SVGError GRL_BMPDEC_clipping_setup ( SVGInt32               dst_area_width,
                                     SVGInt32               dst_area_height,
                                     SVGInt32               *p_dst_xpos,
                                     SVGInt32               *p_dst_ypos,
                                     SVGBmpDecRect          *clip_rect,
                                     SVGUint16              src_width,
                                     SVGUint16              src_height,
                                     SVGBoolean             *p_is_clipped )
                              
{
    SVGError    ret_err         = SVG_NO_ERROR;

	SVG_BMP_U("ENTER INTO GRL_BMPDEC_CLIPPING_SETUP ");

    *p_is_clipped = SVG_FALSE;

    /* Bitmap is outside the destination to the left */
    if ( *p_dst_xpos < 0 )
    {
        clip_rect->x     -= (U16)*p_dst_xpos;
        clip_rect->width += (U16)*p_dst_xpos;
        *p_dst_xpos = 0;
        *p_is_clipped = SVG_TRUE;
    }

    /* Bitmap is outside the destination to the right */
    if ( (*p_dst_xpos + clip_rect->width) > dst_area_width )
    {
        clip_rect->width = (U16)(dst_area_width - *p_dst_xpos);
        *p_is_clipped     = SVG_TRUE;
    }

    /* Bitmap is outside the destination to the top */
    if ( *p_dst_ypos < 0 )
    {
        clip_rect->y      -= (U16)*p_dst_ypos;
        clip_rect->height += (U16)*p_dst_ypos;
        *p_dst_ypos = 0;
        *p_is_clipped = SVG_TRUE;
    }

    /* Bitmap is outside the destination to the bottom */
    if ( (*p_dst_ypos + clip_rect->height) > dst_area_height )
    {
        clip_rect->height = (U16)(dst_area_height - *p_dst_ypos);
        *p_is_clipped = SVG_TRUE;
    }

    /* If bitmap is clipped make sure that flag is set */
    if ( (0 != clip_rect->x) || (0 != clip_rect->y) ||
         (dst_area_width != clip_rect->width) || (dst_area_height != clip_rect->height) ||
         (src_width != clip_rect->width) || (src_height != clip_rect->height))
    {
        *p_is_clipped = SVG_TRUE;
    }

	SVG_BMP_U("EXIT FROM GRL_BMPDEC_CLIPPING_SETUP ");

    return ret_err;
}


SVGError GRL_BMPDEC_read_conf( void )
{
    SVGError                    ret_err         = SVG_NO_ERROR;
    SVGError                    conf_err        = SVG_NO_ERROR;

    /* Get info whether to support JPEG */
    conf_err = GRL_get_int_config( GRL_BMPDEC_DEVKEY_USE_JPEG, &(grl_bmpdec_conf.use_jpeg), 1 );
    if ( SVG_NO_ERROR > conf_err )
    {
        ret_err = conf_err;
    	SVG_BMP_W("GRL_BMPDEC_READ_CONF : Could not read devconf entry USE_JPEG, using default value");
        grl_bmpdec_conf.use_jpeg = GRL_BMPDEC_DEFAULT_USE_JPEG;
    }
    if ( (grl_bmpdec_conf.use_jpeg < 0 ) ||
         (grl_bmpdec_conf.use_jpeg > GRL_BMPDEC_MAX_USE_JPEG) )
    {
    	SVG_BMP_W("GRL_BMPDEC_READ_CONF : Devconf entry USE_JPEG out of range, using default value");
        grl_bmpdec_conf.use_jpeg = GRL_BMPDEC_DEFAULT_USE_JPEG;
    }

    /* Get info whether to support PNG */
    conf_err = GRL_get_int_config( GRL_BMPDEC_DEVKEY_USE_PNG, &(grl_bmpdec_conf.use_png), 1 );
    if ( SVG_NO_ERROR > conf_err )
    {
        ret_err = conf_err;
    	SVG_BMP_W("GRL_BMPDEC_READ_CONF : Could not read devconf entry USE_PNG, using default value.");
        grl_bmpdec_conf.use_png = GRL_BMPDEC_DEFAULT_USE_PNG;
    }
    if ( (grl_bmpdec_conf.use_png < 0 ) ||
         (grl_bmpdec_conf.use_png > GRL_BMPDEC_MAX_USE_PNG) )
    {
    	SVG_BMP_W("GRL_BMPDEC_READ_CONF : Devconf entry USE_PNG out of range, using default value.");
        grl_bmpdec_conf.use_png = GRL_BMPDEC_DEFAULT_USE_PNG;
    }

    /* Get PNG display gamma */
    conf_err = GRL_get_int_config( GRL_BMPDEC_DEVKEY_PNG_DISP_GAMMA, &(grl_bmpdec_conf.png_disp_gamma), 1 );
    if ( SVG_NO_ERROR > conf_err )
    {
        ret_err = conf_err;
        SVG_BMP_W("GRL_BMPDEC_READ_CONF : Could not read devconf entry DISP_GAMMA, using default value.");
        grl_bmpdec_conf.png_disp_gamma = GRL_BMPDEC_DEFAULT_DISP_GAMMA;
    }

    /* Get PNG file gamma */
    conf_err = GRL_get_int_config( GRL_BMPDEC_DEVKEY_PNG_FILE_GAMMA, &(grl_bmpdec_conf.png_file_gamma), 1 );
    if ( SVG_NO_ERROR > conf_err )
    {
        ret_err = conf_err;
        SVG_BMP_W("GRL_BMPDEC_READ_CONF : Could not read devconf entry FILE_GAMMA, using default value.");
        grl_bmpdec_conf.png_file_gamma = GRL_BMPDEC_DEFAULT_FILE_GAMMA;
    }

    /* Get decoding queue size */
    conf_err = GRL_get_int_config( GRL_BMPDEC_DEVKEY_QUEUE_SIZE, &(grl_bmpdec_conf.queue_size), 1 );
    if ( SVG_NO_ERROR > conf_err )
    {
        ret_err = conf_err;
        SVG_BMP_W("GRL_BMPDEC_READ_CONF : Could not read devconf entry QUEUE_SIZE, using default value.");
        grl_bmpdec_conf.queue_size = GRL_BMPDEC_DEFAULT_QUEUE_SIZE;
    }
    if ( (grl_bmpdec_conf.queue_size < GRL_BMPDEC_MIN_QUEUE_SIZE) ||
         (grl_bmpdec_conf.queue_size > GRL_BMPDEC_MAX_QUEUE_SIZE) )
    {
    	SVG_BMP_W("GRL_BMPDEC_READ_CONF : Devconf entry QUEUE_SIZE out of range, using default value.");
        grl_bmpdec_conf.queue_size = GRL_BMPDEC_DEFAULT_QUEUE_SIZE;
    }

    /* Get decoding task priority */
    conf_err = GRL_get_int_config( GRL_BMPDEC_DEVKEY_DEC_TSK_PRIO, &(grl_bmpdec_conf.dec_tsk_prio), 1 );
    if ( SVG_NO_ERROR > conf_err )
    {
        ret_err = conf_err;
        SVG_BMP_W("GRL_BMPDEC_READ_CONF : Could not read devconf entry TSK_PRIO, using default value.");
        grl_bmpdec_conf.dec_tsk_prio = GRL_BMPDEC_DEFAULT_TSK_PRIO;
    }
    if ( (grl_bmpdec_conf.dec_tsk_prio < GRL_BMPDEC_MIN_TSK_PRIO) ||
         (grl_bmpdec_conf.dec_tsk_prio > GRL_BMPDEC_MAX_TSK_PRIO) )
    {
        /* Invalid devconf value, output warning and use default */
    	SVG_BMP_W("GRL_BMPDEC_READ_CONF : Devconf entry TSK_PRIO out of range, using default value.");
        grl_bmpdec_conf.dec_tsk_prio = GRL_BMPDEC_DEFAULT_TSK_PRIO;
    }

    /* Get decoding task stack size */
    conf_err = GRL_get_int_config( GRL_BMPDEC_DEVKEY_DEC_TSK_STKSZ, &(grl_bmpdec_conf.dec_tsk_stksz), 1 );
    if ( SVG_NO_ERROR > conf_err )
    {
        ret_err = conf_err;
        SVG_BMP_W("GRL_BMPDEC_READ_CONF : Could not read devconf entry TSK_STKSZ, using default value.");
        grl_bmpdec_conf.dec_tsk_stksz = GRL_BMPDEC_DEFAULT_TSK_STKSZ;
    }
    if ( (grl_bmpdec_conf.dec_tsk_stksz < GRL_BMPDEC_MIN_TSK_STKSZ) ||
         (grl_bmpdec_conf.dec_tsk_stksz > GRL_BMPDEC_MAX_TSK_STKSZ) )
    {
        /* Invalid devconf value, output warning and use default */
    	SVG_BMP_W("GRL_BMPDEC_READ_CONF :Devconf entry TSK_STKSZ out of range, using default value.");
        grl_bmpdec_conf.dec_tsk_stksz = GRL_BMPDEC_DEFAULT_TSK_STKSZ;
    }

    /* Get max number of contexts */
    conf_err = GRL_get_int_config( GRL_BMPDEC_DEVKEY_CTX_MAX, (SVGInt32*)&(grl_bmpdec_conf.ctx_max), 1 );
    if ( SVG_NO_ERROR > conf_err )
    {
        ret_err = conf_err;
        SVG_BMP_W("GRL_BMPDEC_READ_CONF :Could not read devconf entry CTX_MAX, using default value.");
        grl_bmpdec_conf.ctx_max = GRL_BMPDEC_DEFAULT_CTX_MAX;
    }
    if ( (grl_bmpdec_conf.ctx_max < GRL_BMPDEC_MIN_CTX_MAX) ||
         (grl_bmpdec_conf.ctx_max > GRL_BMPDEC_MAX_CTX_MAX) )
    {
        /* Invalid devconf value, output warning and use default */
    	SVG_BMP_W("GRL_BMPDEC_READ_CONF :Devconf entry CTX_MAX out of range, using default value.");
        grl_bmpdec_conf.ctx_max = GRL_BMPDEC_DEFAULT_CTX_MAX;
    }
    
    /* Get info whether to support FGS */
	conf_err = GRL_get_int_config( GRL_BMPDEC_DEVKEY_USE_FGS, &(grl_bmpdec_conf.use_fgs), 1 );
	if ( SVG_NO_ERROR > conf_err )
	{
	   ret_err = conf_err;
	   SVG_BMP_W("GRL_BMPDEC_READ_CONF :Devconf entry USE_JPEG out of range, using default value.");
	   grl_bmpdec_conf.use_fgs = GRL_BMPDEC_DEFAULT_USE_FGS;
	}
	if ( (grl_bmpdec_conf.use_fgs < 0 ) ||
		(grl_bmpdec_conf.use_fgs > GRL_BMPDEC_MAX_USE_FGS) )
	{
		SVG_BMP_W("GRL_BMPDEC_READ_CONF :Devconf entry TSK_PRIO out of range, using default value.");
	   grl_bmpdec_conf.use_fgs = GRL_BMPDEC_DEFAULT_USE_FGS;
	}

    /* Get info whether to convert RGB to BGR internally during decoding */
	conf_err = GRL_get_int_config( GRL_BMPDEC_DEVKEY_RGB_TO_BGR, &(grl_bmpdec_conf.rgb_to_bgr), 1 );
	if ( SVG_NO_ERROR > conf_err )
	{
	   ret_err = conf_err;
	   SVG_BMP_W("GRL_BMPDEC_READ_CONF :Devconf entry USE_JPEG out of range, using default value.");
	   grl_bmpdec_conf.rgb_to_bgr = GRL_BMPDEC_DEFAULT_RGB_TO_BGR;
	}
	if ( (grl_bmpdec_conf.rgb_to_bgr < GRL_BMPDEC_MIN_RGB_TO_BGR ) ||
		(grl_bmpdec_conf.rgb_to_bgr > GRL_BMPDEC_MAX_RGB_TO_BGR) )
	{
		SVG_BMP_W("GRL_BMPDEC_READ_CONF :Devconf entry TSK_PRIO out of range, using default value.");
	   grl_bmpdec_conf.rgb_to_bgr = GRL_BMPDEC_DEFAULT_RGB_TO_BGR;
	}
    return ret_err;
}


SVGError GRL_BMPDEC_get_conf( GRL_BMPDEC_conf *config )
{
    SVGError    ret_err         = SVG_NO_ERROR;
    
    if ( NULL != config )
    {
        *config = grl_bmpdec_conf;
    }
    else
    {
        ret_err = SVG_POINTER_NULL;
    	SVG_BMP_E("SVG_POINTER_NULL IN GRL_BMPDEC_GET_CONF");
    }

    return ret_err;
}
